import { style } from "./style.js?revision=22410";
import { localeEn, localeRu } from "./locale.js?revision=22410";
import { components } from "./components/components.js?revision=22410";
import useApiTabWazzupExtensionStore from "./store/useApiTabWazzupExtensionStore.js";
import useAccessTabWazzupExtensionStore from "./store/useAccessTabWazzupExtensionStore.js";
import useChannelsTabWazzupExtensionStore from "./store/useChannelsTabWazzupExtensionStore.js";
import useAdditionallyTabWazzupExtensionStore from "./store/useAdditionallyTabWazzupExtensionStore.js";

export default function(settings, metadata) {
  // console.log(settings);
  // console.log(metadata);

  let app = Vue.createApp({
    setup() {
      const tab = Vue.ref('api');
      Vue.provide('metaGroups', metadata.groups || []);
      Vue.provide('metaUsers', metadata.users || []);
      Vue.provide('metaTables', metadata.tables || []);
      Vue.provide('metaFields', metadata.fields || []);
      Vue.provide('metaFilters', metadata.filters || []);

      const { t } = VueI18n.useI18n();
      const apiStore = useApiTabWazzupExtensionStore();
      const accessStore = useAccessTabWazzupExtensionStore();
      const channelStore = useChannelsTabWazzupExtensionStore();
      const additionallyStore = useAdditionallyTabWazzupExtensionStore();

      function getLocaleStorageData() {
        let data = JSON.parse(localStorage.getItem('wazzup_chat'));
        if (data === null) {
          data = {
            show: false,
            position: { top: '10%', left: '10%' },
            size: { height: '600px', width: '500px' },
            tab: 'api',
          };
        }
        return data;
      }

      const setTab = (value) => {
        if (apiStore.api_key == '') {
          value = 'api';
        }
        tab.value = value;
        let currentData = getLocaleStorageData();
        localStorage.setItem(`wazzup_chat`, JSON.stringify({...currentData, tab: value}));
      }

      Vue.onMounted(() => {
        apiStore.api_key = settings?.api_key || "";
        accessStore.groups = settings?.groups || [];
        channelStore.channels = settings?.channels || [];

        additionallyStore.maximum_number_of_messages_per_hour =
          settings?.maximum_number_of_messages_per_hour || '60';

        additionallyStore.def_chan_for_reminder =
          settings?.def_chan_for_reminder || additionallyStore.getFirstChannelId;

        setTab(getLocaleStorageData()['tab'] ?? 'api');

        accessStore.users = (function(users) {
          let temp = [];
          Object.keys(users).forEach(idx => {
            temp.push(users[idx]);
          });
          return temp;
        })(settings?.users || []);

        apiStore.webhook_uri = (function(uri) {
          if (uri.includes('/api/dev/cbext/clientbase/wazzup/webhook')) {
            return uri.split('/')[2];
          }
          return uri;
        })(settings?.webhook_uri || "");

        if (apiStore.api_key !== "") {
          apiStore.authRequest();
        }
      });

      return {
        tab,
        setTab,

        save() {
          fetch('./api/dev/cbext/clientbase/wazzup/save', {
            method: "POST",
            credentials: 'include',
            headers: {
              'X-Auth-Token': CB.globals.x_auth_token,
              'Content-Type': 'application/vnd.api+json',
            },
            body: JSON.stringify({
              api_key: apiStore.api_key,
              webhook_uri: apiStore.webhook_uri,
              groups: accessStore.groups,
              users: accessStore.users,
              channels: channelStore.channels,
              def_chan_for_reminder: additionallyStore.def_chan_for_reminder,
              maximum_number_of_messages_per_hour: additionallyStore.maximum_number_of_messages_per_hour,
            }),
          })
          .then(response => response.text())
          .then(result => {
            if (result === 'saved') {
              window.displayNotification(t('settings_saved'), 1);
              location.reload();
            } else {
              console.log(result); // принтуем ошибку в консоль
              window.displayNotification(t('settings_not_saved'), 2);
            }
          });
        }
      }
    },

    template: /*html*/`
      <div class="navigation-panel">
        <template v-for="item in ['api', 'channels', 'access', 'additionally']">
          <span
            :class="{'navigation-panel__active': (tab == item)}"
            @click="setTab(item)"
          >{{ $t(item + '_tab') }}</span>
          <span v-if="item != 'additionally'">/</span>
        </template>
      </div>

      <component
        :is="tab + '-component'"
        @save-handle="save"
      ></component>
    `,
  });

  CB.styles.globed(style);
  CB.styles.inject();

  components.forEach(component => app.component(component.name, component));

  app.use(Pinia.createPinia());
  app.use(VueI18n.createI18n({
    locale: (CB.globals.config.lang == "russian" ? "ru" : "en"),
    fallbackLocale: 'en',
    messages: {
      en: localeEn,
      ru: localeRu,
    },
  }));
  app.mount("#wazzup");
}
