/**
 * Мини-приложение Быстрые Таблицы КБ
 *
 * Отработка взаимодействия vue-фронта на PrimeVue с jsonapi-беком КБ
 *
 * Базовый компонент - формирование и отображения списка функциональных блоков
 *
 */

import localeEn from './locale/en.js'
import localeRu from './locale/ru.js'
import { useBlocksStore } from './storeBlocks.js'
import { useRecordsStore } from './storeRecords.js'
import { useRecordEditorStore } from './storeRecordEditor.js'
import toshgen from './services/toshgen.js'
import blockText from './components/blockText.js'
import blockTable from './components/blockTable.js'
import blockTextSettings from './components/blockTextSettings.js'
import blockTableSettings from './components/blockTableSettings.js'
import RecordEditor from './components/RecordEditor.js'


const template = /*html*/`
<p-toolbar>
  <template #start>
    {{ $t("message.main_title") }}
  </template>
  <template #end>
    <p-button label="Add Text" @click="op_toggle($event,'cb-block-text-settings')" class="mr-2"></p-button>
    <p-button label="Add Table" @click="op_toggle($event,'cb-block-table-settings')" class="mr-2"></p-button>
  </template>
</p-toolbar>
<p-overlaypanel ref="op" append-to="body" :dismissable="true" style="width: 300px">
  <component :is="op_component"></component>
  <p-button label="Create" @click="op_create" class="mr-2"></p-button>
  <p-button label="Cancel" @click="op_hide" class="mr-2"></p-button>
</p-overlaypanel>
<div class="card">
  <component v-for="item in blockList"  :is="item.block" :key="item.id" :blockid="item.id"></component>
</div>
<p-sidebar v-model:visible="isVisibleRecordEditor" position="right" class="p-sidebar-lg">
	<cb-record-editor></cb-record-editor>
</p-sidebar>
<!--
<pre v-pre>{{ blockList }}</pre>
<pre v-pre>{{ knowledge }}</pre>
-->
`;

CB.styles.globed(/*css*/`
#fastabs_app {
  margin: 0rem 2rem;
}
`);

const fastabsI18n = VueI18n.createI18n({
  locale: 'ru',
  fallbackLocale: 'en',
  messages: {
    en: localeEn,
    ru: localeRu,
  },
});

const FastabsApp = {
  template,
  components: {
    "cb-block-text": blockText,
    "cb-block-text-settings": blockTextSettings,
    "cb-block-table": blockTable,
    "cb-block-table-settings": blockTableSettings,
    "cb-record-editor": RecordEditor,
    "p-slider": primevue.slider,
    "p-tabmenu": primevue.tabmenu,
    "p-button": primevue.button,
    "p-toolbar": primevue.toolbar,
    "p-panel": primevue.panel,
    "p-overlaypanel": primevue.overlaypanel,
    "p-sidebar": primevue.sidebar,
  },
  setup() {
    const blocksStore = useBlocksStore()
    const { blockList, new_block } = Pinia.storeToRefs(blocksStore)
    const { isVisibleRecordEditor } = Pinia.storeToRefs(useRecordEditorStore())

    const op = Vue.ref()
    const op_component = Vue.ref('')

    function defaultBlockSettings(block) {
      switch (block) {
        case 'cb-block-table-settings': return { block: 'cb-block-table', title: 'Table', table_id: 70, }
        default: return { block: 'cb-block-text', title: 'Text', content: toshgen(25), }
      }
    }

    const op_toggle = (event, component) => {
      new_block.value = defaultBlockSettings(component)
      op_component.value = component
      op.value.toggle(event)
    };

    const op_create = () => {
      blocksStore.addBlock()
      op.value.hide()
    };

    const op_hide = () => {
      op.value.hide()
    };

    return {
      blockList,
      op,
      op_component,
      op_toggle,
      op_create,
      op_hide,
      isVisibleRecordEditor,
    }
  }
}

Vue.createApp(FastabsApp)
  .use(Pinia.createPinia())
  .use(fastabsI18n)
  .use(primevue.config.default)
  .mount('#fastabs_app')
