import { useEditFormStore } from '../../../store/storeForm.js';

const { defineAsyncComponent } = Vue;

const ObjectsList = {
  name: 'objects-list',

  components: {
    // Асинхронная регистрация, чтобы разорвать цикл
    'edit-form-field': defineAsyncComponent(() =>
      import("../../entities/FormLine/FormField/FormField.js")
    ),
  },

  props: { 
    storeFieldId: String, 
    dataSource: Object, 
    fieldsSource: Object, 
    size: String, 
    width: String 
  },

  data() {
    return  { 
      mode: '', 
      formStatus: '', 
      editIndex: -1, 
      form: {}, 
      tablesDict: {},
      ablesDictLoaded: false,
    };
  },

  mounted() {
    if (!Array.isArray(this.dataSource[this.storeFieldId])) {
      this.dataSource[this.storeFieldId] = [];
    }
  },

  computed: {
    field() { return this.fieldsSource?.[this.storeFieldId] || {}; },
    nestedFields() {
      const f = this.field?.fields || {};
      if (Array.isArray(f)) {
        const map = {};
        f.forEach(id => { map[id] = this.fieldsSource?.[id] || { type: 'input-text' }; });
        return map;
      }
      return f;
    },
    list() {
      return Array.isArray(this.dataSource[this.storeFieldId]) ? this.dataSource[this.storeFieldId] : [];
    },
    titleField() {
      const entries = Object.entries(this.nestedFields);
      const byFlag = entries.find(([, cfg]) => !!cfg.is_list_name);
      return byFlag ? byFlag[0] : (entries[0]?.[0] || null);
    },
    addText() { return this.field?.addtext || 'add_table'; },
  },

  methods: {
    async handleButtonAction() {
      const cfg = this.nestedFields.button;
      if (!cfg) return;

      // если в button задан apiAction — выполняем запрос
      if (cfg.apiAction) {
        try {
          let response = await fetch("." + cfg.apiAction, {
            method: "GET",
            credentials: "include",
            headers: {
              'X-Auth-Token': window.x_auth_token || null,
              'Content-Type': 'application/vnd.api+json'
            }
          });
          response = await response.json();
          if (response['module-settings'] && this.store?.setData) {
            this.store.setData(response['module-settings']);
          }
          let msg = response[cfg.messageIndex || 'message'] || 'OK';
          jalert(msg, '', '', () => window.location.reload());
        } catch (e) {
          console.error('Ошибка при выполнении кнопки:', e);
        }
      } else {
        // если apiAction не задан — просто вывести в консоль
        console.log('Кнопка нажата', cfg);
      }
    },
    async ensureTablesDict() {
      if (this.tablesDictLoaded) return;

      const meta = window.CB && window.CB.metadata;
      if (!meta || typeof meta.getDataTables !== 'function') return; // мета ещё не готова

      try {
        // как делает FormField: передаём isRoot()
        const { useEditFormStore } = await import('../../../store/storeForm.js');
        const store = useEditFormStore();
        const list = await meta.getDataTables(store.isRoot());

        // нормализуем в {id:name} независимо от формата
        const map = {};
        if (Array.isArray(list)) {
          if (list.length && Array.isArray(list[0])) {
            // [[id, name], ...]
            for (const [id, name] of list) map[String(id)] = name;
          } else {
            // [{id, name}, ...] (или похожий объект со свойствами)
            for (const t of list) {
              const id = t.id ?? t[0];
              const name = t.name ?? t[1];
              if (id != null && name != null) map[String(id)] = name;
            }
          }
        } else if (list && typeof list === 'object') {
          // { id: name, ... }
          for (const [id, name] of Object.entries(list)) map[String(id)] = name;
        }

        this.tablesDict = map;
        this.tablesDictLoaded = true;
      } catch (e) {
        console.warn('tablesDict load failed:', e);
      }
    },
    openAdd() {
      this.mode = 'add'; this.editIndex = -1;
      const obj = {};
      for (const [id, cfg] of Object.entries(this.nestedFields)) {
        // никаких автозначений для knowledge_table_id
        if (cfg.default !== undefined) obj[id] = cfg.default;
        if (cfg.type === 'checkbox' && !Array.isArray(obj[id])) obj[id] = [];
      }
      this.form = obj;
    },

    openEdit(i) {
      this.mode = 'edit'; this.editIndex = i;
      this.form = JSON.parse(JSON.stringify(this.list[i] || {}));
      this.ensureTablesDict();
    },
    cancel() { 
      this.mode = ''; 
      this.editIndex = -1; 
      this.form = {}; 
    },
    save() {
      if (!Array.isArray(this.dataSource[this.storeFieldId])) this.dataSource[this.storeFieldId] = [];
      if (this.mode === 'add') this.dataSource[this.storeFieldId].push({ ...this.form });
      else if (this.mode === 'edit' && this.editIndex > -1) this.dataSource[this.storeFieldId][this.editIndex] = { ...this.form };
      this.cancel();
    },
    remove(i) { 
      if (!Array.isArray(this.dataSource[this.storeFieldId])) return; 
      this.dataSource[this.storeFieldId].splice(i, 1); 
      if (this.mode==='edit'&&this.editIndex===i) this.cancel(); 
    },
    generateNewName() {
      const prefix = 'Таблица';
      let n = this.list.length + 1;
      const exists = v => this.list.some(it => String(it[this.titleField] || '') === v);
      while (exists(`${prefix} ${n}`)) n++;
      return `${prefix} ${n}`;
    },
    fieldList() { 
      return Object.keys(this.nestedFields); 
    },
    getItemTitle(item, index) {
      if (!this.titleField) return '#' + (index + 1);
      const cfg = this.nestedFields[this.titleField] || {};
      const raw = item[this.titleField];

      // статические options
      if (cfg.options && cfg.options[raw] != null) return cfg.options[raw];

      // специальный случай — выпадашка таблиц
      if (cfg.optionsFrom && cfg.optionsFrom.source === 'tables') {
        // запускаем ленивую подгрузку; без await — ререндер случится, когда словарь приедет
        this.ensureTablesDict();
        const name = this.tablesDict[String(raw)];
        if (name) return name;
      }

      // запасной вариант
      return raw ?? ('#' + (index + 1));
    },
     hasValidTable(parentFieldId) {
      const val = this.form[parentFieldId];
      if (val == null || val === '') return false;
      // принимаем числовые id и строковые числа ('42')
      if (typeof val === 'number') return true;
      if (/^\d+$/.test(String(val))) return true;
      // если вдруг пришло имя — не считаем валидным (чтобы не дергать API)
      return false;
    },
    showField(fid) {
      const cfg = this.nestedFields[fid] || {};

      // стандартная логика showOn
      if (cfg.showOn) {
        const v = this.form[cfg.showOn];
        const ok = Array.isArray(cfg.showOnValue)
          ? cfg.showOnValue.includes(v)
          : v === cfg.showOnValue;
        if (!ok) return false;
      }

      // спец-случай: поле с optionsFrom: fields — только когда выбран валидный parent
      if (cfg.optionsFrom?.source === 'fields') {
        const parent = cfg.optionsFrom.parent;
        if (!this.hasValidTable(parent)) return false;
      }

      return true;
    },
  },

  template: `
    <section class="objects-list">
      <ul v-if="list.length" class="objects-list__items" style="list-style:none; padding:0; margin:8px 0 6px;">
        <li v-for="(item,i) in list" :key="i" class="objects-list__item"
            style="display:flex; align-items:center; gap:6px; margin:4px 0;">
          <!-- кнопка удаления с иконкой -->
          <button class="delete-button_2bnmabnygj"
                  title="Удалить"
                  style="background:none; border:none; padding:0; cursor:pointer;"
                  @click.prevent="remove(i)">
            <img src="images/b_drop.png" alt="Удалить" />
          </button>

          <a href="#" @click.prevent="openEdit(i)"
            style="color:#2a5db0; text-decoration:underline; font-size:13px;">
            {{ getItemTitle(item, i) }}
          </a>
        </li>
      </ul>

      <a v-if="!mode"
        href="#"
        @click.prevent="openAdd"
        style="display:inline-block; margin-top:6px; color:#2a5db0; text-decoration:underline; font-size:13px;">
        {{ $t(addText) }}
      </a>

      <form v-if="mode" class="objects-list__form" @submit.prevent="save"
          style="margin-top:10px; display:flex; flex-direction:column; gap:10px;">

        <div class="objects-list__grid"
            style="display:flex; flex-direction:column; gap:8px;">
          <edit-form-field
            v-for="fid in fieldList()"
            :key="fid"
            :storeFieldId="fid"
            :dataSource="form"
            :fieldsSource="nestedFields"
          />
        </div>

        <div class="objects-list__buttons"
            style="display:flex; gap:10px; align-items:center; margin-top:8px;">
          <button type="submit" class="btn btn-default">
            {{ $t(mode === 'edit' ? 'editform_button_save' : 'editform_button_add') }}
          </button>

          <button type="button" class="btn btn-default" @click.prevent="cancel">
            {{ $t('editform_button_cancel') }}
          </button>

          <button
            v-if="nestedFields.button && nestedFields.button.type === 'button'"
            type="button"
            class="btn btn-default"
            @click.prevent="handleButtonAction"
          >
            {{ $t(nestedFields.button.text || nestedFields.button.label || 'button') }}
          </button>
        </div>
      </form>
    </section>
  `
};

export default ObjectsList;
