import {IRequestData} from './table';
import TableView from './table-view';
import App from '../app';
import {findAncestor, URL, getUrlVars, KeyCode, renderPage} from '../util';
import Model from '../model';
import Access from '../access';

export default class TableController {
    private tableModel: Model;

    public init(data: IRequestData = {
        pagination: 0,
        search: ``
    }): void {
        if (typeof data.pagination === `undefined`) {
            data.pagination = 0;
        }

        if (typeof data.search === `undefined`) {
            data.search = ``;
        }

        this.tableModel = new Model();
        this.tableModel.getData(`${URL}?pagination=${data.pagination}`)
            .then((response: any): void => { // Отправляем запрос на получение доступа к поиску
                Access.setAccess(response);
                const searchParam = (Access.search === 1) ? `&search=${data.search}` : ``;
                this.tableModel.getData(`${URL}?pagination=${data.pagination}${searchParam}`)
                    .then((response: any): void => { // Отправляем запрос для получения данных и отображения
                        const tableView: TableView = new TableView(response);
                        renderPage(tableView);
                        if (Access.show === 1) {
                            const table = document.querySelector(`#cb-table-wrap`);

                            if (table) {
                                table.classList.add(`cb-table-wrap--show`);
                            }
                        }
                        this.setSearch();
                        this.bind();
                    })
                    .catch((err: Error): void => {
                        throw new Error(`Error with render table, ${err}`);
                    });
            });
    }

    private setSearch = (): void => {
        const searchInput: HTMLInputElement = document.querySelector(`#search-input`);

        if (searchInput && getUrlVars()[`search`]) {
            searchInput.value = decodeURI(getUrlVars()[`search`]);
        }
    }

    private onPagClick = (e: Event) => {
        this.bind(false);
        const pag: any = e.target;
        const index: number = parseInt(pag.getAttribute(`index`));
        const search: string = getUrlVars()[`search`] || ``;
        App.showTable({
            pagination: index,
            search: search
        });
    }

    private onRowClick = (e: Event) => {
        e.stopPropagation();
        const target: any = e.target;
        if (target.tagName !== `A`) {
            const row: HTMLElement = findAncestor(target, `cb-table__row`);

            if (row) {
                const id = parseInt(row.getAttribute(`row-id`));
                if (id) {
                    this.bind(false);
                    App.showLine({
                        id: id
                    });
                }
            }
        }
    }

    private onSearchSubmit = (): void => {
        const searchByInput: HTMLInputElement = document.querySelector(`#search-input`);
        this.searchByInput(searchByInput);
    }

    private onSearchKeyDown = (e: Event|any): void => {
        if (e.keyCode === KeyCode.ENTER) {
            this.searchByInput(e.target);
        }
    }

    private searchByInput = (searchInput: HTMLInputElement|any): void => {
        if (searchInput) {
            this.bind(false);
            const val = searchInput.value;

            if ((typeof getUrlVars()[`search`] !== `undefined` && val !== getUrlVars()[`search`]) || val !== ``) {
                App.showTable({
                    pagination: 0,
                    search: val
                });
            }
        }
    }

    private onAddNewClick = (e: Event): void => {
        this.bind(false);
        App.showEdit({
            id: 0
        });
    }

    private bind = (bind: boolean = true): void => {
        const pags: any = document.querySelectorAll(`.cb-pagination__btn`);

        if (pags.length > 0) {
            for (let pag of pags) {
                if (bind) {
                    pag.addEventListener(`click`, this.onPagClick);
                } else {
                    pag.removeEventListener(`click`, this.onPagClick);
                }
            }
        }

        if (Access.show === 1) {
            const rows: any = document.querySelectorAll(`.cb-table__row`);

            if (rows.length > 0) {
                for (let row of rows) {
                    if (bind) {
                        row.addEventListener(`click`, this.onRowClick);
                    } else {
                        row.removeEventListener(`click`, this.onRowClick);
                    }
                }
            }
        }

        const searchSubmit: HTMLButtonElement = document.querySelector(`#search-submit`);

        if (searchSubmit) {
            if (bind) {
                searchSubmit.addEventListener(`click`, this.onSearchSubmit);
            } else {
                searchSubmit.removeEventListener(`click`, this.onSearchSubmit);
            }
        }

        const searchInput: HTMLInputElement = document.querySelector(`#search-input`);

        if (searchInput) {
            if (bind) {
                searchInput.addEventListener(`keydown`, this.onSearchKeyDown);
            } else {
                searchInput.removeEventListener(`keydown`, this.onSearchKeyDown);
            }
        }

        const addNewLine: HTMLButtonElement = document.querySelector(`#add-line`);

        if (addNewLine) {
            if (bind) {
                addNewLine.addEventListener(`click`, this.onAddNewClick);
            } else {
                addNewLine.removeEventListener(`click`, this.onAddNewClick);
            }
        }
    }
}