//Отображение режима
function view_mode(mode, type) {
    if (mode === undefined) {
        mode = 0;
    }
    if (type === undefined) {
        type = 'calc';
    }
    mode = Number(mode);
    if (mode === 0) {
        $('[data-type=construct]').show();
        $('[data-type=expert]').hide();
        var status = $('input[name="use_params"][value="1"]').prop('checked');
        if (status) {
            view_params(1);
        } else {
            view_params(0);
        }
    } else {
        $('[data-type=construct]').hide();
        $('[data-type=expert]').show();
    }
}

function view_params(status) {
    if (status === undefined) {
        status = 0;
    }
    status = Number(status)
    if (status === 0) {
        $('#calc_params_block').hide();
        $('#calc_params_block_create').hide();
        $('.show_size_div').hide();
    } else {
        $('#calc_params_block').show();
        $('#calc_params_block_create').show();
        $('.show_size_div').show();
    }
}
/**
 * Создает и возвращает иконку добавления в виде элемента `<img>`.
 *
 * @param {Object} params - Параметры для создания иконки.
 * @param {Function} params.onClick - Обработчик события `click` для иконки.
 *
 * @returns {jQuery} - Элемент jQuery, представляющий иконку добавления.
 *
 * @description
 * - Создает элемент `<img>` с изображением "green_plus.png".
 * - Применяет класс `icon icon_create` для оформления.
 * - Устанавливает обработчик события `click`, переданный в параметрах.
 *
 * @example
 * const addIcon = createAddIcon({ onClick: () => console.log('Icon clicked!') });
 * $('body').append(addIcon);
 */
const createAddIcon = ({ onClick }) => $("<img>")
  .attr({
    "src": "images/green_plus.png",
    "class": "icon icon_create"
  })
  .click(onClick);

/**
 * Создает и возвращает элемент `<img>` для значка удаления с указанными атрибутами и обработчиком клика.
 *
 * @param {Object} params - Параметры для создания элемента `<img>`.
 * @param {string | number} params.dataTplNum - Номер шаблона, устанавливаемый в атрибут `data-tpl-num`.
 * @param {Function} params.onClick - Обработчик события `click`, который вызывается при клике на значок.
 *
 * @returns {jQuery} - Элемент jQuery, представляющий созданный значок удаления.
 *
 * @description
 * - Устанавливает атрибуты `src`, `data-tpl-num` и `class` для элемента `<img>`.
 * - Присоединяет обработчик события `click`.
 *
 * @example
 * const deleteIcon = createDeleteIcon({
 *   dataTplNum: 3,
 *   onClick: () => console.log('Delete clicked!')
 * });
 * $('body').append(deleteIcon);
 */
const createDeleteIcon = ({
  dataTplNum,
  onClick
}) => 
  $("<img>")
    .attr({
      "src": "images/b_drop.png",
      "data-tpl-num": dataTplNum,
      "class": "icon icon_delete"
    })
    .click(onClick);

/**
 * Создает и возвращает элемент `<img>` для значка редактирования с указанными атрибутами и обработчиком клика.
 *
 * @param {Object} params - Параметры для создания элемента `<img>`.
 * @param {string | number} params.dataTplNum - Номер шаблона, устанавливаемый в атрибут `data-tpl-num`.
 * @param {Function} params.onClick - Обработчик события `click`, который вызывается при клике на значок.
 *
 * @returns {jQuery} - Элемент jQuery, представляющий созданный значок редактирования.
 *
 * @description
 * - Устанавливает атрибуты `src`, `data-tpl-num` и `class` для элемента `<img>`.
 * - Присоединяет обработчик события `click`.
 *
 * @example
 * const editIcon = createEditIcon({
 *   dataTplNum: 5,
 *   onClick: () => console.log('Edit clicked!')
 * });
 * $('body').append(editIcon);
 */
const createEditIcon = ({
  dataTplNum,
  onClick
}) => 
  $("<img>")
    .attr({
      "src": "images/b_edit.png",
      "data-tpl-num": dataTplNum,
      "class": "icon icon_edit"
    })
    .click(onClick);


/**
 * Создает и возвращает строку (`<div>` элемент) с указанными дочерними элементами.
 *
 * @param {Object} params - Параметры для создания строки.
 * @param {Array<jQuery | HTMLElement | null>} params.children - Массив дочерних элементов, которые будут добавлены в строку.
 *   - `null` или `undefined` элементы будут пропущены.
 *
 * @returns {jQuery} - Элемент jQuery, представляющий строку с дочерними элементами.
 *
 * @description
 * - Создает `<div>` с классом `construct_select`.
 * - Добавляет каждый непустой дочерний элемент из `children` внутрь созданного `<div>`.
 *
 * @example
 * const row = createRow({
 *   children: [$('<span>').text('Child 1'), $('<span>').text('Child 2')]
 * });
 * $('body').append(row);
 */
const createRow = ({
  children,
}) => {
  const $row = $("<div>")
    .attr({class: "construct_select"})

  children.forEach(child => {
    if(child)
      $row.append(child);
  });
  return $row;
}
/**
 * Создает и возвращает элемент `<select>` с указанными атрибутами, дочерними элементами и обработчиком события изменения.
 *
 * @param {Object} params - Параметры для создания элемента `<select>`.
 * @param {string | number} params.dataTableId - Идентификатор таблицы, устанавливаемый в `data-table_id`.
 * @param {string | number} params.tplId - Значение шаблона, устанавливаемое в `value`.
 * @param {string | number} params.dataTplNum - Номер шаблона, устанавливаемый в атрибут `data-tpl-num` и в `name`.
 * @param {Array<jQuery | HTMLElement>} params.children - Массив дочерних элементов (`<option>`), которые будут добавлены внутрь `<select>`.
 * @param {Function} params.onChange - Обработчик события `change`, который вызывается при изменении значения в `<select>`.
 *
 * @returns {jQuery} - Элемент jQuery, представляющий созданный `<select>`.
 *
 * @description
 * - Устанавливает класс `form-control form-control-250 select_tpl` и другие атрибуты, такие как `name`, `data-tpl-num`, `data-table_id` и `value`.
 * - Присоединяет обработчик события изменения.
 * - Добавляет переданные дочерние элементы внутрь созданного элемента `<select>`.
 *
 * @example
 * const select = createTemplateSelect({
 *   dataTableId: 1,
 *   tplId: "template1",
 *   dataTplNum: 3,
 *   children: [$('<option>').text('Option 1')],
 *   onChange: () => console.log('Changed!')
 * });
 * $('body').append(select);
 */
const createTemplateSelect = ({
  dataTableId,
  tplId,
  dataTplNum,
  children,
  onChange,
}) => {

  const $templateSelect = 
    $("<select>")
    .attr({
      name: "construct_tpl[" + dataTplNum + "]",
      "data-tpl-num": dataTplNum,
      "data-table_id": dataTableId,
      class: "form-control form-control-250 select_tpl",
      value: tplId
    })
  $templateSelect.change(onChange);
  $templateSelect.append(children);

  return $templateSelect;
}
/**
 * Удаляет элементы шаблона с указанным ключом и очищает связанные события.
 *
 * @param {string | number} tplKey - Ключ шаблона, который используется для выбора элементов с атрибутом `data-tpl-num`.
 *
 * @returns {void}
 *
 * @description
 * - Ищет элементы с атрибутом `data-tpl-num`, равным указанному `tplKey`.
 * - Удаляет найденные элементы из DOM, если они существуют.
 * - Очищает массив событий в объекте `tpl_events` для указанного ключа шаблона.
 *
 * @example
 * deleteTplItem(5); // Удаляет элементы с атрибутом data-tpl-num="5" и очищает связанные события.
 */    
const deleteTplItem = (tplKey) => {
  const $tplItems = $("[data-tpl-num=" + tplKey + "]");
  if ($tplItems.length === 0)
    return;
  
  $tplItems.remove();
  
  tpl_events[tplKey] = [];
}

/**
 * Отображает элементы шаблона с указанным номером и выделяет соответствующий селектор.
 *
 * @param {number | string} tplNum - Номер шаблона для отображения элементов и выделения селектора.
 *
 * @returns {void}
 *
 * @description
 * - Сбрасывает цвет границы всех селекторов с классом `select_tpl` на стандартный.
 * - Устанавливает зеленую границу для селектора, соответствующего переданному номеру `tplNum`.
 * - Скрывает все элементы с классом `td_tpl`.
 * - Показывает только те элементы `td_tpl`, которые имеют атрибут `data-tpl-num`, равный переданному `tplNum`.
 * - Вызывает функцию `bind_help_bt` для привязки событий или инициализации помощи.
 *
 * @example
 * showTpl(3); // Показывает элементы шаблона с номером 3 и выделяет соответствующий селектор.
 */
function showTpl(tplNum) {
  $("select.select_tpl").css('border-color', 'rgb(204, 204, 204)');
  $("select.select_tpl[data-tpl-num='" + tplNum + "']").css('border', '1px solid green');
  $(".td_tpl").hide();
  $("td.td_tpl[data-tpl-num='" + tplNum + "']").show();
  bind_help_bt();
}

/**
 * Создает строку с выпадающим списком шаблонов и соответствующими действиями.
 *
 * @param {Object} params - Параметры для создания строки.
 * @param {string} params.calcType - Тип вычисления, определяющий логику формирования данных.
 * @param {Function} params.onSelectChange - Обработчик изменения значения в выпадающем списке.
 * @param {string | number} params.tplKey - Уникальный ключ шаблона.
 * @param {string | number} params.tableId - Идентификатор таблицы.
 * @param {string | number} params.tplId - Идентификатор шаблона.
 * @param {string | number} params.calcId - Идентификатор вычисления.
 * @param {string} params.type - Тип шаблона.
 * @param {boolean} [params.isDeletable=true] - Указывает, можно ли удалить шаблон.
 * @param {Array} [params.children=[]] - Дополнительные элементы, добавляемые в строку.
 *
 * @returns {jQuery} - Строка с выпадающим списком шаблонов и действиями (редактировать, удалить). 
 * @see {@link createEditIcon} - Создание значка редактирования.
 * @see {@link createDeleteIcon} - Создание значка удаления.
 * @see {@link createTemplateSelect} - Генерация выпадающего списка для шаблона.
 * @see {@link createRow} - Оборачивает элементы в строку.
 * @see {@link get_tpl} - Инициализирует и отображает шаблон.
 * @see {@link get_tpl_sel_items} - Возвращает элементы для выпадающего списка.
 * @see {@link deleteTplItem} - Удаляет шаблон из интерфейса.
 * @see {@link showTpl} - Показывает выбранный шаблон.
 *
 * @description
 * - Формирует выпадающий список с данными для указанного шаблона.
 * - Добавляет значок редактирования, который выделяет и показывает связанные элементы шаблона.
 * - Добавляет значок удаления, если `isDeletable` равно `true`.
 * - Вызывает `get_tpl` для инициализации шаблона.
 * - Объединяет выпадающий список, значки действий и дополнительные элементы в строку.
 *
 * @example
 * const $row = createTemplateSelectRow({
 *   calcType: 'simple',
 *   onSelectChange: handleChange,
 *   tplKey: 1,
 *   tableId: 123,
 *   tplId: 456,
 *   calcId: 789,
 *   type: 'default',
 *   isDeletable: true,
 *   children: [$("<span>").text("Дополнительно")]
 * });
 */
const createTemplateSelectRow = ({
  calcType,
  onSelectChange, 
  tplKey,
  tableId,
  tplId,
  calcId,
  type,
  isDeletable = true,
  children = [],
}) => {
  const data = get_tpl_sel_items(tableId, calcId, type, tplId, calcType);

  const $editIcon = createEditIcon({
    dataTplNum: tplKey,
    onClick: () => showTpl(tplKey)
  });
  const $deleteIcon = isDeletable ? createDeleteIcon({
    dataTplNum: tplKey,
    onClick: () => deleteTplItem(tplKey)
  }): undefined;

  const $templateSelect = createTemplateSelect({ 
    dataTableId: tableId, 
    tplId, 
    dataTplNum: tplKey,
    children: data,
    onChange: onSelectChange,
  });
  
  const $wrapper = createRow({ 
    children: [
      $templateSelect, 
      $editIcon, 
      $deleteIcon,
      ...children
    ]
  });
  //Вызываем get_tpl, чтобы сформировать шаблон, если он есть
  get_tpl($templateSelect, calcId, type);
  return $wrapper;
}

/**
 * Создает первую строку для выбора шаблона с кнопкой создания нового шаблона.
 * 
 * @param {Object} params - Объект параметров.
 * @param {Function} params.onCreateButtonClick - Обработчик клика по кнопке создания шаблона.
 * @param {Object} createTemplateSelectRowProps - Дополнительные параметры для создания строки шаблона.
 * 
 * @returns {jQuery} - Строка HTML элементов для отображения шаблона с кнопкой создания.
 * 
 * @see {@link createAddIcon} - Генерирует и возвращает иконку для создания нового элемента.
 * @see {@link createTemplateSelectRow} - Формирует строку для отображения шаблона.
 */

const createTemplateSelectFirstRow = ({
  onCreateButtonClick,
  ...createTemplateSelectRowProps
}) => {
  const $createIcon = createAddIcon({ onClick: onCreateButtonClick })
  const $row = createTemplateSelectRow({
    ...createTemplateSelectRowProps,
    tplKey: 0, 
    isDeletable: false,
    children: [$createIcon]
  })
  return $row;
}

/**
 * Создает и отображает шаблоны для конструктора на основе переданных параметров.
 * 
 * @param {Object} params - Объект параметров.
 * @param {jQuery} params.$parent - Родительский элемент, в который будут добавляться шаблоны.
 * @param {string} params.tableId - Идентификатор таблицы для привязки шаблонов.
 * @param {string} params.calcId - Идентификатор вычисления.
 * @param {string} params.type - Тип шаблона.
 * @param {Object} params.params - Дополнительные параметры, содержащие идентификаторы шаблонов.
 * @param {string} params.calcType - Тип вычислений для шаблонов.
 * 
 * @returns {void}
 * 
 * @description
 * - Инициализирует создание и отображение шаблонов.
 * - Позволяет добавлять новые шаблоны в интерфейс.
 * - Привязывает обработчики событий для выбора и создания шаблонов.
 * 
 * @see {@link createTemplateSelectRow} - Создаёт строку для выбора шаблона.
 * @see {@link createTemplateSelectFirstRow} - Создаёт первую строку для выбора шаблона.
 * @see {@link get_tpl} - Загружает данные для выбранного шаблона.
 * @see {@link change_tpl} - Переменная для отслеживания изменений в интерфейсе.
 * @see {@link init_chosen} - Инициализирует выбранный элемент интерфейса.
 */
const createConstructorTemplates = ({
  $parent, 
  tableId, 
  calcId, 
  type, 
  params, 
  calcType
}) => {
  const [ getLastTplIndex, setLastTplIndex ] = useState(0);
  
  const handleTemplateSelect = (event) => {
    get_tpl($(event.target), calcId, type);
    change_tpl = 1;
    init_chosen();
  }

  const addNewTpl = () => {
    setLastTplIndex(getLastTplIndex() + 1);
    const newTplNum = getLastTplIndex();

    const $row = createTemplateSelectRow({
      tplKey: newTplNum, 
      tableId,
      tplId: 0,
      calcId,
      type,
      onSelectChange: handleTemplateSelect,
      calcType,
    })
    
    $parent.append($row);
  }

  if (!params || !params['id']) {
    const $row = createTemplateSelectFirstRow({
      tableId,
      tplId: 0,
      calcId,
      type,
      onSelectChange: handleTemplateSelect,
      onCreateButtonClick: addNewTpl,
      calcType,
    })
    $parent.append($row);
    return;
  }

  const { id: templateIds } = params;
  for (const [key, id] of Object.entries(templateIds)) {
    if (Number(key) > getLastTplIndex()) {
      setLastTplIndex(Number(key));
    }
    if(key === '0') {
      const $row = createTemplateSelectFirstRow({
        tableId,
        tplId: id,
        calcId,
        type,
        onSelectChange: handleTemplateSelect,
        onCreateButtonClick: addNewTpl,
        calcType,
      })
      $parent.append($row);
      continue;
    }

    const $row = createTemplateSelectRow({
      tplKey: key, 
      tableId,
      tplId: id,
      calcId,
      onSelectChange: handleTemplateSelect,
      calcType,
      type,
    })
    $parent.append($row);
  }
}

/**
 * Помещает шаблон tpl на страниу 
 * @param {*} data - шаблон
 * @param {*} num_tpl - номер шаблона
 */
function load_tpl(data, num_tpl) {
    var calc_tpl = $("#calc_tpl_tr");
    var td_tpl = $("td.td_tpl[data-tpl-num='" + num_tpl + "']");
    $("select.select_tpl").css('border-color', 'rgb(204, 204, 204)');
    $("select.select_tpl[data-tpl-num='" + num_tpl + "']").css('border', '1px solid green');
    $(".td_tpl").hide();
    /**
     * Это объект состояния в глобальной видмости 
     * Создается в edit_calc.tpl
     * Хранит информацию о полях измененных 
     * при взаимодействии с пользоватем
     * Обрабатывается в edit_calc.tpl,
     * чтобы вызвать динамику в шаблоне edit_calc.tpl
     * 
     * Прим.: мне пришлось, было гораздо хуже
     */
    if (typeof setEventFields === "function") 
        setStartCalcFields([]); // Сбрасываем список полей, чтобы edit_calc.tpl почистил динамику
    if (td_tpl.length == 0) {
        // первичный рендер шаблона
        var td_tpl = $("<td>")
            .attr({
                "data-tpl-num": num_tpl,
                "colspan": 2,
                "align": "center"
            })
            .addClass("td_tpl")
            .append(data);
        calc_tpl.append(td_tpl);
    } else {
        // ререндер шаблона
        td_tpl.html(data);
        td_tpl.show();
    }
    //bind_help_bt();
    if (calc_tpl.css('display') === 'none') {
        calc_tpl.show();
    }
    var correlation = new Correlation(num_tpl);
    correlation.init();
    bind_help_bt();
    setTimeout(() => {
        init_chosen();
    }, 100);
}

function expert_mode_button(id) {
    jconfirm('Запуск в режиме эксперта приведет к потере возможности редактирования в режиме конструктора, вы уверены?',
        function () {
            yes_cons();
        },
        function () {
            no_cons();
        }
    );

    function yes_cons() {
        $.ajax({
            type: 'POST',
            url: 'calc_construct.php',
            data: 'resetConstruct=' + id + '&csrf=' + csrf,
            success: function () {
                window.location.reload();
            }
        });
        return false;
    }

    function no_cons() {
        return false;
    }
}



function date_cond(cond, date2) {
    if (cond.val() === 'period') {
        date2.show();
    } else {
        date2.hide();
    }
}

function setLink(option, input) {
    if (option !== undefined) {
        input.val(option.data('link'));
    }
}

function open_parameters() {

}

function cond_mode() {
    var conds_mode = $('input[name="set_conds"]:checked');
    var conds_block = $('#cond_block');
    if (conds_mode === undefined) {
        $('input[name="set_conds"][value="0"]').prop('checked', 'true');
    } else {
        var status = Number(conds_mode.val());
        if (status) {
            conds_block.show();
        } else {
            conds_block.hide();
        }
    }
}

/* AJAX запросы */

//Получает список полей
function getField(table_id, select, name_param, tplKey) {
    table_id = Number(table_id);
    if (table_id !== 0) {
        $.ajax({
            type: 'POST',
            url: 'calc_construct.php',
            data: 'getField=1&csrf=' + csrf + '&name=' + name_param + '&table=' + table_id+ '&tpl_num=' + tplKey,
            success: function (data) {
                if (data) {
                    var info = JSON.parse(data);
                    select.html(info.fields);
                    select.prop('disabled', false);
                    var tplKey = select.data('tpl-num');
                    var param_name = select.data('name');
                    var hidden = $('[name="calc_params[' + tplKey + '][' + param_name + '][field-type]"]');
                    if (hidden.length > 0) {
                        setTypeToHidden(select, tplKey, hidden);
                    }
                }
            }
        });
    } else {
        select.html('');
        select.prop('disabled', true);
    }
}

//Получает поля подтаблицы
function getSubField(table_id, sub_table_id, select, name_param, link_input) {
    if (table_id && sub_table_id) {
        $.ajax({
            type: 'POST',
            url: 'calc_construct.php',
            data: 'getSubField=1&csrf=' + csrf + '&sub_table=' + sub_table_id + '&name=' + name_param + '&table=' + table_id,
            success: function (data) {
                if (data) {
                    var info = JSON.parse(data);
                    select.html(info.fields);
                    select.prop('disabled', false);
                    link_input.val(info.link_field_id);
                }
            }
        });
    } else {
        select.html('');
        select.prop('disabled', true);
    }
}

//Получает список шаблонов
function get_tpl_sel_items(table, calcId, type, selected, calcType) {
    if (selected === undefined) {
        selected = 0;
    }
    if (window.tpl_sel_items !== undefined && calcId === -1) {
        return window.tpl_sel_items;
    } else {
        //Запрашиваем список шаблонов
        var action = 'action=getTpls&ajax';
        window.tpl_sel_items = $.ajax({
            type: 'POST',
            url: 'edit_calc.php',
            async: false,
            data: action + '&csrf=' + csrf + '&table=' + table + '&calc_id=' + calcId + '&type=' + type + '&selected=' + selected + '&calc_type=' + calcType
        }).responseText;
        return window.tpl_sel_items;
    }
}

//Получает шаблон
function get_tpl(sel, calcId, type) {
    let table_id = sel.attr("data-table_id");
    let num_tpl = sel.attr("data-tpl-num");
    let tpl_id = sel.val();
    let select_val = sel.val();

    if (calcId === undefined) {
        calcId = 0;
    }
    if (tpl_id > 0) {
        //Запрашиваем шаблон
        let action = 'action=getTpl&tpl_id=' + tpl_id + '&ajax';

        $.ajax({
            type: 'POST',
            url: 'edit_calc.php',
            data: action + '&csrf=' + csrf + '&table=' + table_id + '&calc_id=' + calcId + '&type=' + type + '&num=' + num_tpl,
            success: function (data) {
                if (data) {
                    load_tpl(data, num_tpl, select_val);
                }
            }
        });
    } else {
        $("td.td_tpl[data-tpl-num='" + num_tpl + "']").remove();
    }
}

function send_ajax(url, params) {
    params.csrf = csrf;
    return JSON.parse($.ajax({
        type: 'POST',
        url: url,
        data: params,
        async: false,
    }).responseText);
}

function setTypeToHidden(obj, tplKey, hidden) {
    var target = $(obj);
    var val = target.val();
    var option = target.find('option[value="' + val + '"]');
    if (option.length > 0 && hidden.length > 0) {
        hidden.val(option.data('type'));
    }
}

function checkVkMessage() {
    $.ajax({ // Формируем запрос
        type: "POST",
        url: "vk_msg_check.php",
        data: {vk_check: 'get', csrf: csrf},
        success: function (msg) {
            if (msg == 'refresh') {
                checkVkMessage();
            } else if (msg == 'error') {
                alert('Error');
            } else if (msg == 'wait') {
                setTimeout(checkVkMessage, 5000);
            }
        }
    });
}
/* END AJAX запросы */

$(document).ready(function () {
    //Переключение режима
    $("input:radio[name=calc_mode]").change(function () {
        view_mode($(this).val());
    });
    $("input:radio[name=use_params]").change(function () {
        view_params($(this).val());
    });
});
