/**
 * Приложение управления API-токенами
 *
 */

import { jsonapi_store, jsonapi_query } from '../jsonapi-basic.js';

const template = /*html*/`
<div align=center>
  <br>{{$t('Tokens_manager')}}<span class='help_bt' h_id='Tokens_manager'></span><br><br>
  <table class='lost_files_table'>
    <tr>
      <th width="20"></th>
      <th width="380">{{$t('Tokens_manager_header_token')}}</th>
      <th width="100">{{$t('Tokens_manager_header_valid_before')}}</th>
      <th width="250">{{$t('Tokens_manager_header_user_name')}}</th>
    </tr>
    <tr v-if="tokens.length == 0">
      <td>&nbsp;</td>
      <td></td>
      <td></td>
      <td></td>
    </tr>
    <tr v-for="item in tokens" :key="item.token">
      <td><img src="images/b_drop.png" style="vertical-align:middle; border:0; cursor:pointer" :alt="$t('delete')" :title="$t('Delete')" @click="deleteToken(item.token)" /></td>
      <td style="font-family:monospace">{{item.token}}</td>
      <td>{{asLocalDate(item.validBefore)}}</td>
      <td>{{item.user_fio}}</td>
    </tr>
  </table>
  <br>
  <p>{{$t('Tokens_manager_add_token')}}</p>
  <table border=0 cellpadding=3>
    <tr>
      <td align=right>{{$t('Tokens_manager_user_id')}}<span class='help_bt' h_id='Tokens_manager_user'></span>:</td>
      <td>
        <select v-model="new_token.user_id" class="form-control form-control-250 chosen_add">
          <option v-for="item in users" :key="item.id" :value="item.id">{{item.fio}}</option>
        </select>
      </td>
    </tr>
    <tr>
      <td align=right>{{$t('Tokens_manager_valid_before')}}<span class='help_bt' h_id='Tokens_manager_valid'></span>:</td>
      <td>
        <cbv-old-date-picker v-model="new_token.validBefore" class="form-control form-control-75"></cbv-old-date-picker>
      </td>
    </tr>
    <tr>
      <td colspan=2 align=center>
        <br /><input type="submit" class="btn btn-default btn-sm" :value="$t('Add')" @click="addNewToken()" />
      </td>
    </tr>
  </table>
  <br>
  <br><a href="edit_add.php" onclick='history.back();return false;'>{{$t('return_to_add_settings')}}</a>
`;

CB.styles.globed(/*css*/`
#tokenier_app {
  /*width: 72rem;
  margin: 5rem auto;*/
}
`);

// Интернационализация, подцепляемся к глобальной lang
const tokenierI18n = VueI18n.createI18n({
  locale: 'ru',
  fallbackLocale: 'en',
  messages: {
    en: lang,
    ru: lang,
  },
});

const tokenierApp = {
  template,
  setup() {
    const users = Vue.ref([]);
    const tokens = Vue.ref([]);
    const new_token = Vue.ref({
      user_d: '',
      validBefore: '',
    });

    /** Загрузить список валидных токенов */
    const getTokens = () => {
      const valid_tokens = jsonapi_store.find('settings', 'core.authmanager.valid_tokens').value ?? {};
      const token_list = [];

      for (const item in valid_tokens) {
        token_list.push({
          token: item,
          user_id: valid_tokens[item].user_id,
          user_fio: 'Loading...',
          validBefore: valid_tokens[item].validBefore,
        });
      }

      tokens.value = token_list;

      // Асинхронно загрузить данные пользователей
      loadUserDetails();
    };

    const loadUserDetails = async () => {
      for (const token of tokens.value) {
        await jsonapi_query(`user/${token.user_id}`);
        let user = jsonapi_store.find('user', token.user_id) ?? {};
        token.user_fio = user.fio;
      }
    };

    getAllActiveUsers();
    jsonapi_query('settings/core.authmanager.valid_tokens').then(() => getTokens());

    /** Получить список валидных токенов как объект для сохранения в БД*/
    const tokensAsPersistObject = () => {
      const result = {};
      tokens.value.forEach(item => {
        result[item.token] = {
          user_id: item.user_id,
          validBefore: item.validBefore,
        }
      })
      return result;
    }

    /** Сохранить список валидных токенов */
    const persistTokenObject = (persistObject) => {
      jsonapi_query('settings/core.authmanager.valid_tokens', {
        method: 'PATCH',
        body: {
          data: {
            type: 'settings',
            id: 'core.authmanager.valid_tokens',
            attributes: {
              value: persistObject,
            }
          }
        },
      })
      .then(() => jsonapi_query('settings/core.authmanager.valid_tokens').then(() => getTokens()));
    }

    /** Создать уникальный идентификатор токена */
    const makeTokenID = (length) => {
      let result = '';
      const characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
      const charactersLength = characters.length;
      for (let i = 0; i < length; i++ ) {
        result += characters.charAt(Math.floor(Math.random() * charactersLength));
      }
      return result;
    }

    /** Конвертировать дату в формате ISO (YYY-MM-DD) в строку соответствии с языковыми настройками пользователя */
    const asLocalDate = (date) => {
      return $.datepicker.formatDate(lang.date_js_format, dayjs(date).toDate());
    }

    /** Добавить новый токен в список валидных */
    const addNewToken = () => {
      if (new_token.value.user_id == '' || new_token.value.validBefore == '') return;
      const date = $.datepicker.parseDate(lang.date_js_format, new_token.value.validBefore);
      if (!date) return;
      date.setHours(12);
      const token = makeTokenID(48);
      const persistObject = tokensAsPersistObject();
      persistObject[token] = {
        user_id: new_token.value.user_id,
        validBefore: date.toISOString().substring(0,10),
      }
      persistTokenObject(persistObject);
      new_token.value = {
        user_d: '',
        validBefore: '',
      }

      setTimeout(() => $("select.chosen_add").trigger("chosen:updated"), 10);
    }

    /** Удалить валидный токен */
    const deleteToken = (token) => {
      const persistObject = tokensAsPersistObject();
      delete persistObject[token];
      persistTokenObject(persistObject);
    }

    /** Рекурсивно получить всех активных пользователей */
    async function getAllActiveUsers(offset = 0) {
      let response = await jsonapi_query(`user?page[limit]=50&page[offset]=${offset}&filter=and(eq(arc, '0'), ne(group_id, '777'))`);
      if (response.data.length > 0) {
        return getAllActiveUsers(offset += 50);
      } else {
        users.value = jsonapi_store.findAll('user');
      }
    }

    // Инициализация chosen с функцией при change
    function initChosen() {
      $('.chosen_add').chosen().change(function(event) {
        new_token.value.user_id = event.target.value;
      });
    }

    Vue.onMounted(() => {
      setTimeout(() => initChosen(), 1000);
    });

    // Инициализация данных
    getAllActiveUsers();
    jsonapi_query('settings/core.authmanager.valid_tokens').then(() => getTokens());

    Vue.watch(users, () => {
      setTimeout(() => $("select.chosen_add").trigger("chosen:updated"), 100);
    });

    return {
      users,
      tokens,
      new_token,
      addNewToken,
      deleteToken,
      asLocalDate,
    }
  }
}

Vue.createApp(tokenierApp)
  .use(tokenierI18n)
  .component('cbv-old-date-picker', {
    template: '<input :value="modelValue" @input="onInput($event.target.value)" />',
    props: [ 'modelValue', 'class' ],
    methods: {
      onInput: function(datestr) {
        this.$emit('update:modelValue', datestr);
      },
    },
    mounted: function() {
      var self = this;
      $(this.$el).addClass(this['class']);
      $(this.$el).datepicker({
        showOn: "button",
        dateFormat: lang.date_js_format,
        buttonImage: "images/calbtn.png",
        buttonImageOnly: true,
        buttonText: "{$lang.Calendar}",
        onSelect: function(datestr) {
          self.$emit('update:modelValue', datestr);
        }
      });
      $('.chosen').chosen()
    },
    beforeDestroy: function() {
      $(this.$el).datepicker('hide').datepicker('destroy');
    }
  })
  .mount('#tokenier_app')
