import columns from "./column.js";

/** Главный компонент */
export default function (json) {
  return {
    name: "kanban-main-component",

    components: {
      columns,
    },

    data() {
      return {
        boards: json,
        len: Object.keys(json?.data || {}).length,
        hide: true,
      };
    },

    mounted() {
      this.syncScrollListeners();
      this.updateScrollWidth();

      window[`updateDataKanban${this.boards.settings.num}`] = this.updateData;

      // Перерасчёт при изменении размеров окна
      window.addEventListener("resize", this.updateScrollWidth);
    },

    methods: {
      getSortedData() {
        let result = {};
        Object.keys(this.boards.data)
          .sort((a, b) => this.boards.data[a].num - this.boards.data[b].num)
          .forEach((item) => (result[item] = this.boards.data[item]));

        return result;
      },

      getTitleHeight() {
        let i = 0,
          w = false;

        for (let key in this.boards.data) {
          let word = this.boards.data[key].title?.value || "";
          let space = word.slice(0, word.indexOf(" ")).length || 0;

          if (space > 13) {
            w = true;
          }

          let cur_i = Math.ceil(
            (this.boards.data[key].title?.value?.length || 0) / 14
          );
          if (cur_i > i) {
            i = cur_i;
          }
        }

        let height = 20 + (i - 1) * 35;
        height = w ? height : height / 2;

        return height < 60 ? 60 : height;
      },

      async updateData() {
        let result = await fetch(
          window.location.href + "&getblock=" + this.boards.settings.num
        );
        result = await result.json();
        this.boards.data = result.data;

        this.len = Object.keys(result.data || {}).length;
        this.$nextTick(() => {
          this.updateScrollWidth();
        });
      },

      syncScroll() {
        const top = this.$refs.scrollTop;
        const bottom = this.$refs.scrollBottom;
        if (top && bottom) {
          top.scrollLeft = bottom.scrollLeft;
        }
      },

      syncScrollListeners() {
        const top = this.$refs.scrollTop;
        const bottom = this.$refs.scrollBottom;

        if (top && bottom) {
          top.addEventListener("scroll", () => {
            bottom.scrollLeft = top.scrollLeft;
          });

          bottom.addEventListener("scroll", () => {
            top.scrollLeft = bottom.scrollLeft;
          });
        }
      },

      updateScrollWidth() {
        this.$nextTick(() => {
          const bottom = this.$refs.scrollBottom;
          const ref = this.$refs.scrollWidthRef;

          if (bottom && ref) {
            ref.style.width = bottom.scrollWidth + "px";
          }
        });
      },
    },

    template: /*html*/ `
      <div>
        <!-- ВЕРХНИЙ СКРОЛЛ -->
        <div
          ref="scrollTop"
          class="kanban-scroll-top"
          style="overflow-x: auto; overflow-y: hidden; height: 16px; margin-bottom: 5px;"
        >
          <div ref="scrollWidthRef" style="height: 1px;"></div>
        </div>

        <!-- Заголовок -->
        <div class="kanban-header" @click="hide = !hide;">
          {{ boards.settings.caption }}
        </div>

        <transition name="fade">
          <!-- ОСНОВНОЙ КОНТЕНТ -->
          <div
            ref="scrollBottom"
            class="kanban-flex-wrap"
            style="display: flex; overflow-x: auto;"
            v-if="hide"
            @scroll="syncScroll"
          >
            <columns
              v-for="column of getSortedData()"
              :key="column.id"
              :column="column"
              :len="len"
              :data="boards.data"
              :settings="boards.settings"
              :kanban_id="boards.kanban_id"
              :title_height="getTitleHeight()"
              @update_data="updateData"
            />
          </div>
        </transition>
      </div>
    `,
  };
}
