(function () {

    /**
     * Словарь для типов полей параметра
     * @const
     * @type {{number}}
     */
    var fieldType = {
        INPUT: 0,
        SELECT: 1,
        CHECKBOX: 2,
        TEXTAREA: 3
    };

    var createType = {
        CREATE: {
            type: 0,
            titleText: 'Добавление нового параметра:'
        },
        EDIT: {
            type: 1,
            titleText: 'Редактирования параметра:'
        }
    };

    /**
     * Данные для поля `Сохранить в поле`
     * после запроса на их получения, массив присваивается в эту переменную
     * @type {Array}
     */
    var fieldSaveData = [];
    var fieldSaveAssocData = [];

    /**
     * Установка атрибутов и параметров в поля параметров
     * @param data {Object}
     * @param index {number}
     * @returns {jquery}
     * @private
     */
    var _setFieldsAttrs = function (data, index) {
        var result = null;
        if (typeof index === 'number'
            && data && typeof data === 'object') {
            var keys = data['fields']['keys'];
            var valFields = data['fields']['valFields'];
            var rows = data['fields']['rows'];
            var nameSpan = $('<span class="calc_params_name">')
                .text(data['paramNameText']);
            keys[index].append(nameSpan)
                .append('<span>:&nbsp;</span>');
            var field;
            if (data['fieldType'] === fieldType.SELECT) {
                field = $('<select>');
            } else if (data['fieldType'] === fieldType.CHECKBOX) {
                field = $('<input>', {
                    type: 'checkbox',
                    css: {
                        'width': '15px',
                        'margin-top': '7px'
                    }
                });
            } else if (data['fieldType'] === fieldType.TEXTAREA) {
                field = $('<textarea>');
            } else {
                field = $('<input>');
            }
            valFields[index].append(field
                .addClass('form-control form-control-100')
                .attr('id', data['id']));
            rows[index]
                .append(keys[index])
                .append(valFields[index]);
            result = rows[index];
        }

        return result;
    };

    /**
     * Выводит фрагмент состоящий из сформированных option`ов для select`а
     * @param data {Object|array} Данные для option`ов
     * @returns {jQuery|HTMLElement}
     * @private
     */
    var _initOptionsForSelect = function (data) {
        var frag = $(document.createDocumentFragment());
        if (data && typeof data === 'object') {
            frag.append('<option>');
            for (var key in data) {
                var fieldData = data[key];
                var option = $('<option>');
                if (typeof fieldData === 'object') {
                    option
                        .attr({
                            'value': fieldData['id'],
                            'data-type': fieldData['type_field']
                        })
                        .text(fieldData['name_field']);
                } else if (typeof fieldData === 'string') {
                    option
                        .attr('value', key)
                        .text(fieldData);
                }
                frag.append(option);
            }
        }

        return frag;
    };

    /**
     * Инициализация поля `Сохранить в поле`
     * @private
     */
    var _initFieldSave = function () {
        if (fieldSaveData && typeof fieldSaveData === 'object') {
            var field = $('#calc-param-field-save');
            if (field.length > 0) {
                var frag = _initOptionsForSelect(fieldSaveData);
                field.append(frag);
            }
        }
    };

    /**
     * Инициализация поля `Из поля`
     * @param xhr {string} Ответ от сервера
     * @private
     */
    var _initFromField = function (xhr) {
        var fromField = $('#calc-param-from-field');
        if (xhr && fromField.length > 0) {
            var data = JSON.parse(xhr);
            var frag = _initOptionsForSelect(data);
            fromField.html('');
            fromField.append(frag);
        }
    };

    /**
     * Инициализация поля `Значение по умолчанию`
     * @param newField {jquery|HTMLElement}
     * @private
     */
    var _initDefaultField = function (newField) {
        if (newField && typeof newField !== 'undefined') {
            var defaultField = $('#calc-param-default-val');
            if (defaultField.length > 0) {
                var parent = defaultField.parents('.calc_params_create_val');
                if (parent.length > 0) {
                    newField.attr({
                        'class': newField.attr('class') + ' form-control form-control-100',
                        'id': defaultField.attr('id'),
                        'autocomplete': 'off'
                    });
                    defaultField
                        .off();
                    parent.html('');
                    parent.append(newField);
                    if (newField.hasClass('datepicker')) {
                        var current_date = $('<a>')
                            .attr({
                                'href': '#'
                            })
                            .click(function () {
                                __setCurrentDate(newField);
                            })
                            .text(lang.current_date);
                        parent.append(current_date);
                        var pickerParams = {
                            showOn: 'button',
                            dateFormat: lang.date_js_format,
                            buttonImage: 'images/calbtn.png',
                            buttonImageOnly: true,
                            buttonText: lang.Calendar
                        };
                        if (newField.hasClass('datetime')) {
                            newField.datetimepicker(pickerParams);
                        } else {
                            newField.datepicker(pickerParams);
                        }
                    } else if (newField.hasClass('combobox')) {
                        var url = 'select_value.php?table=' + this.table_id + '&field=' + $('#calc-param-field-save').val();
                        create_combobox(newField);
                        autocomplete_ajax_request(newField, url, false, false, undefined, '', {}, false);
                    }
                }
            }
        }
    };

    /**
     * Заполнение поля `Значение по умолчанию` значением {current}
     * @param field {jquery|HTMLElement}
     * @private
     */
    var __setCurrentDate = function (field) {
        field.val('{current}');
    };

    /**
     * Заполнение поля `Значение по умолчанию` option`ами, если поле = список
     * @param xhr
     * @private
     */
    var _initDefaultSelectField = function (xhr) {
        var defaultField = $('#calc-param-default-val');
        if (xhr && typeof xhr === 'string' && defaultField.length > 0) {
            var data = JSON.parse(xhr);
            var frag = _initOptionsForSelect(data);
            defaultField.html('')
                .append(frag);
        }
    };

    /**
     * Валидация на непустоту имени параметра
     * @returns {boolean}
     * @private
     */
    var _validateByEmptyName = function () {
        var nameField = $('#calc-param-name');
        var result = false;
        if (nameField.length > 0) {
            if (nameField.val().trim() !== '') {
                result = true;
            }
        }
        if (!result) {
            jalert(lang['Fill_param_name']);
        }

        return result;
    };

    /**
     * Валидация на непустоту сохранить в поле
     * @returns {boolean}
     * @private
     */
    var _validateByEmptySaveField = function () {
        var nameField = $('#calc-param-field-save');
        var result = false;
        if (nameField.length > 0) {
            if (nameField.val() > 0) {
                result = true;
            }
        }
        if (!result) {
            jalert(lang['Fill_save_field']);
        }

        return result;
    };

    /**
     * Валидация на индивидуальное имя параметра
     * @param data {Object|array} Данные других параметров
     * @returns {boolean}
     * @private
     */
    var _validateByDuplicateName = function (data) {
        var result = false;
        if (data && typeof data === 'object') {
            var nameField = $('#calc-param-name');
            if (nameField.length > 0) {
                result = true;
                var name = nameField.val();
                var saveBtn = $('#calc_params_save-btn');
                var paramId = saveBtn.data('id');
                for (var paramKey in data) {
                    var paramData = data[paramKey];
                    if ((name === paramData['name'])
                        && ((typeof paramId === 'undefined')
                            || (typeof paramId === 'number'
                                && paramId !== parseInt(paramKey)))) {
                        result = false;
                    }
                }
            }
        }
        if (!result) {
            jalert(lang['Duplicate_param_name']);
        }

        return result;
    };

    /**
     * Валидация параметров
     * @param data
     * @returns {boolean}
     * @private
     */
    var _validateParamsProps = function (data) {
        var result = false;
        if (data && typeof data === 'object') {
            if (Object.keys(data).length > 0) {
                result = _validateByEmptyName();
                if (result) {
                    result = _validateByDuplicateName(data);
                }
                if (result) {
                    result = _validateByEmptySaveField();
                }
            } else {
                result = _validateByEmptyName();
                if (result) {
                    result = _validateByEmptySaveField();
                }
            }
        }
        return result;
    };

    /**
     * Создание полей заполнения для параметра
     * type {object} Тип создания полей
     * @private
     */
    var _createParamsFields = function (type, id) {
        var paramsFieldsWrap = $('.calc_params_create-div');
        if (paramsFieldsWrap.length > 0 && type && typeof type === 'object') {
            paramsFieldsWrap.html('');
            var frag = $(document.createDocumentFragment());
            var title = $('<span>').text(type['titleText']);
            frag.append(title);
            var rows = [],
                keys = [],
                valFields = [];
            for (var i = 0; i < 6; i++) {
                rows[i] = $('<div class="calc_params_create_row">');
                keys[i] = $('<div class="calc_params_create_key">');
                valFields[i] = $('<div class="calc_params_create_val">');
            }
            var fields = {
                rows: rows,
                keys: keys,
                valFields: valFields
            };
            /** Название параметра */
            rows[0] = _setFieldsAttrs({
                fields: fields,
                paramNameText: 'Название',
                id: 'calc-param-name',
                fieldType: fieldType.INPUT
            }, 0);

            /** Сохранить в поле */
            rows[1] = _setFieldsAttrs({
                fields: fields,
                paramNameText: 'Сохранить в поле',
                id: 'calc-param-field-save',
                fieldType: fieldType.SELECT
            }, 1);

            /** Значение по умолчанию */
            rows[2] = _setFieldsAttrs({
                fields: fields,
                paramNameText: 'Значение по умолчанию',
                id: 'calc-param-default-val',
                fieldType: fieldType.INPUT
            }, 2);

            /** Из поля */
            rows[3] = _setFieldsAttrs({
                fields: fields,
                paramNameText: 'Из поля',
                id: 'calc-param-from-field',
                fieldType: fieldType.SELECT
            }, 3);

            /** Редактирование */
            rows[4] = _setFieldsAttrs({
                fields: fields,
                paramNameText: 'Не редактируемое',
                id: 'calc-param-not-edit',
                fieldType: fieldType.CHECKBOX
            }, 4);

            /** Обязательно для заполнения */
            rows[5] = _setFieldsAttrs({
                fields: fields,
                paramNameText: lang.Required_param,
                id: 'calc-param-required',
                fieldType: fieldType.CHECKBOX
            }, 5);

            rows.forEach(function (row) {
                frag.append(row);
            });
            var saveBtn = $('<button>')
                .attr({
                    'class': 'btn btn-default btn-sm calc_params_save-btn',
                    'id': 'calc_params_save-btn',
                    'type': 'button',
                    'data-type': type['type']
                })
                .text('Сохранить параметр');
            if (typeof id !== 'undefined') {
                saveBtn.attr('data-id', id);
            }
            frag.append(saveBtn);
            paramsFieldsWrap.append(frag)
                .show();
            _initFieldSave();
        }
    };

    /**
     *
     * @constructor
     * @augments EditButton
     */
    var ParamsConstruct = function () {
        this.paramsCount = 0;
    };

    /***
     * Наследование от EditButton
     * @type {EditButton}
     */
    ParamsConstruct.prototype = Object.create(EditButton.prototype);

    /**
     * Заполнение полей параметра данными
     * @param data {Object|array} данные о параметре
     */
    ParamsConstruct.prototype.initParamsFields = function (data) {
        if (data && typeof data !== 'undefined') {
            let fieldSave = $('#calc-param-field-save');
            let nameField = $('#calc-param-name');
            let defaultVal = $('#calc-param-default-val');
            let fromField = $('#calc-param-from-field');
            let notEdit = $('#calc-param-not-edit');
            let requiredParam = $('#calc-param-required');

            if (fieldSave.length > 0
                && nameField.length > 0
                && defaultVal.length > 0
                && fromField.length > 0) {
                if (typeof data['name'] === 'string') {
                    nameField.val(data['name']);
                }
                if (typeof data['field'] !== 'undefined') {
                    fieldSave.val(data['field']);
                    fieldSave.change(_onFieldSaveChange.bind(this));
                    this.getFromFieldData({
                        fieldType: data['type_field'],
                        fieldId: data['field']
                    });
                    var defaultField = this.getFieldByType(data['type_field'], fieldSaveAssocData[data['field']]);
                    _initDefaultField(defaultField);
                    if (defaultField[0].nodeName === 'SELECT' && !defaultField.hasClass('combobox')) {
                        this.getDefaultFieldData(data['field']);
                    }
                }
                if (typeof data['default_value']['value'] === 'string') {
                    defaultVal = $('#calc-param-default-val');
                    if (data['default_value']['is_field'] === true || data['default_value']['is_field'] === 1) {
                        fromField.val(data['default_value']['value']);
                    } else {
                        defaultVal.val(data['default_value']['value']);
                        if (defaultVal.hasClass('combobox')) {
                            setValueToAc(defaultVal);
                        }
                    }
                }
                if (typeof data['not_edit'] !== 'undefined') {
                    data['not_edit'] = Number(data['not_edit']);
                    if (data['not_edit'] === 1) {
                        notEdit.prop('checked', true);
                    } else {
                        notEdit.prop('checked', false);
                    }
                }
                if (typeof data['required'] !== 'undefined') {
                    data['required'] = Number(data['required']);
                    if (data['required'] === 1) {
                        requiredParam.prop('checked', true);
                    } else {
                        requiredParam.prop('checked', false);
                    }
                }
            }
            var saveChange = $('#calc_params_save-btn');
            if (saveChange.length > 0) {
                saveChange.click(_onSaveBtnClick.bind(this));
            }
        }

        setTimeout(init_chosen, 0);
    };

    /**
     * Отрисовка нового параметра
     * @param id {string|number} id параметра
     * @param data {Object|array} Данные параметра
     */
    ParamsConstruct.prototype.renderNewParam = function (id, data) {
        if (typeof id === 'undefined') {
            id = this.paramsCount;
        }
        if (typeof data === 'undefined') {
            data = this.paramsData[this.paramsCount];
        }
        var dropBtn = $('<button class="cond-remove-btn btn-drop">')
            .attr({
                'type': 'button',
                'data-id': id
            })
            .click(_onDropParamClick.bind(this));
        var paramLink = $('<a class="calc_param_link">')
            .attr('data-id', id)
            .text(data['name'])
            .click(_onParamLinkClick.bind(this));
        var paramBlock = $('<div class="calc_params_block">')
            .attr({
                'id': 'calc_params_block_' + id,
                'data-id': id
            });
        paramBlock
            .append(dropBtn)
            .append(paramLink);
        $('.calc_params_list').append(paramBlock);
    };

    /**
     * Измненения параметра
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.changeParam = function (id) {
        var nameField = $('#calc-param-name');
        var link = $('.calc_param_link[data-id="' + id + '"]');
        if (nameField.length > 0 && link.length > 0) {
            link.text(nameField.val());
        }
    };

    /**
     * Сохранение параметра
     * @param type {number} Тип сохранения
     * 0 - при добавлении параметра
     * 1 - при редактировании параметра
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.saveParam = function (type, id) {
        let fieldSave = $('#calc-param-field-save');
        let nameField = $('#calc-param-name');
        let defaultVal = $('#calc-param-default-val');
        let fromField = $('#calc-param-from-field');
        let notEdit = $('#calc-param-not-edit');
        let requiredParam = $('#calc-param-required');

        if (fieldSave.length > 0
            && nameField.length > 0
            && defaultVal.length > 0
            && fromField.length > 0
            && requiredParam.length > 0) {
            let typeField = 0;
            let isField = 0;
            let fieldVal = '';
            let notEditVal = 0;
            let isRequired = 0;

            if (fieldSave.val()) {
                typeField = fieldSave.find('option[value=' + fieldSave.val() + ']').data('type');
            }
            if (fromField.val()) {
                isField = 1;
                fieldVal = fromField.val();
            } else {
                fieldVal = defaultVal.val();
            }
            if (typeof id === 'undefined') {
                id = this.paramsCount;
            }
            if (notEdit.prop('checked')) {
                notEditVal = 1;
            }
            if (requiredParam.prop('checked')) {
                isRequired = 1;
            }
            var data = {
                'name': nameField.val(),
                'field': fieldSave.val(),
                'type_field': typeField,
                'default_value': {
                    'is_field': isField,
                    'value': fieldVal
                },
                'not_edit': notEditVal,
                'required': isRequired
            };
            if (this.buttonId === -1) {
                this.paramsData[id] = data
            }
            if (type === createType.CREATE['type']) {
                if (this.buttonId === -1) {
                    this.renderNewParam();
                    this.paramsCount++;
                } else {
                    this.saveParamData(data);
                }
            } else if (type === createType.EDIT['type']) {
                if (this.buttonId === -1) {
                    this.changeParam(id);
                } else {
                    this.saveParamData(data, id);
                }
            }
            $('.calc_params_create-div').html('');
            $('#calc_params_create-btn').show();
            $('.calc_no_params').hide();
        }
    };

    /** Event`s callbacks */

    /**
     * @event e
     * @private
     */
    var _onSaveBtnClick = function (e) {
        var correct = false;
        var target = $(e.target);
        var type = target.data('type');
        correct = _validateParamsProps(this.paramsData);
        if (correct) {
            if (type === createType.EDIT['type']) {
                var id = target.data('id');
                this.saveParam(type, id);
            } else {
                this.saveParam(type);
            }

        }
    };

    /**
     * @event e
     * @private
     */
    var _onFieldSaveChange = function (e) {
        var target = $(e.target);
        var val = target.val();
        var fieldType;
        if (val) {
            fieldType = target.find('option[value=' + val + ']').data('type');
        } else {
            fieldType = '';
        }

        this.getFromFieldData({
            fieldType: fieldType,
            fieldId: val
        });
        var parameters;
        parameters = Number(val) !== 0 ? fieldSaveAssocData[val] : 0;
        var defaultField = this.getFieldByType(fieldType, parameters);
        _initDefaultField(defaultField);
        if (defaultField[0].nodeName === 'SELECT' && !defaultField.hasClass('combobox')) {
            this.getDefaultFieldData(val);
        }

        setTimeout(init_chosen, 0);
    };

    ParamsConstruct.prototype.setParamName = function () {
        var paramName = $('#calc-param-name');
        var paramLinks = $("a.calc_param_link[data-id]");
        var name = 'Параметр 1';
        if (paramName.length > 0 && !paramName.val()) {
            if (paramLinks.length > 0) {
                var num = 0;
                var temp = 0;
                $.each( paramLinks, function( key, value ) {
                    if (value.text.indexOf("Параметр") + 1) {
                        temp = value.innerText.match(/Параметр[' ']+[\d\w]+/);
                        if (!temp) {
                            temp = num+1;
                        } else {
                            temp = temp[0].split(" ");
                            temp = Number(temp[1]);
                        }
                        if (num < temp) {
                            num = temp;
                        }
                    }
                });
                num++;
                name = 'Параметр ' + num;
            }
            paramName.val(name);
        }
    };
    /**
     * @event e
     * @private
     */
    var _onAddParamClick = function (e) {
        _createParamsFields(createType.CREATE);
        this.setParamName();
        $(e.target).hide();
        var fieldSave = $('#calc-param-field-save');
        if (fieldSave.length > 0) {
            fieldSave.change(_onFieldSaveChange.bind(this));
        }
        var saveParamBtn = $('#calc_params_save-btn');
        if (saveParamBtn.length > 0) {
            saveParamBtn.click(_onSaveBtnClick.bind(this));
        }

        setTimeout(init_chosen, 0);
    };

    /**
     * @event e
     * @private
     */
    var _onParamLinkClick = function (e) {
        var target = $(e.target);
        var id = target.data('id');
        _createParamsFields(createType.EDIT, id);
        if (this.buttonId === -1) {
            this.initParamsFields(this.paramsData[id]);
        } else {
            this.getParamsFields(id);
        }
        $('#calc_params_create-btn').show();

        setTimeout(init_chosen, 0);
    };

    /**
     * @event e
     * @private
     */
    var _onDropParamClick = function (e) {
        var target = $(e.target);
        var id = target.data('id');
        this.dropParam(id);
    };

    /** ================= */

    /** Getters data by ajax */

    /**
     * Получение данных о параметра
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.getParamsFields = function (id) {
        $.ajax({
            url: this.ajaxUrl,
            data: {
                ajax: 1,
                method: 'GET',
                action: 'get',
                button_id: this.buttonId,
                param_id: id
            },
            success: function (xhr) {
                if (xhr) {
                    this.initParamsFields(JSON.parse(xhr));
                }
            }.bind(this)
        });
    };

    /**
     * Получение данных для поля `Сохранить в поле`
     */
    ParamsConstruct.prototype.getFieldSaveData = function () {
      if (isNaN(this.tableId)) {
        this.tableId = parseInt(_getDefaultParams('table-id'));
        this.buttonId = parseInt(_getDefaultParams('button-id')) || -1;
      }

        $.ajax({
            url: this.ajaxUrl,
            method: 'GET',
            data: {
                ajax: 1,
                action: 'getAllFields',
                table_id: this.tableId,
                button_id: this.buttonId
            },
            success: function (xhr) {
                if (xhr) {
                    data = JSON.parse(xhr);
                    fieldSaveData = data['not_assoc'];
                    fieldSaveAssocData = data['assoc'];
                }
            }
        });
    };

    /**
     * Получение данных для поля `Из поля`
     * @param data
     */
    ParamsConstruct.prototype.getFromFieldData = function (data) {
      if (isNaN(this.tableId)) {
        this.tableId = parseInt(_getDefaultParams('table-id'));
        this.buttonId = parseInt(_getDefaultParams('button-id')) || -1;
      }

        if (data && typeof data === 'object') {
            $.ajax({
                url: this.ajaxUrl,
                method: 'GET',
                async: false,
                data: {
                    ajax: 1,
                    action: 'getFields',
                    table_id: this.tableId,
                    button_id: this.buttonId,
                    type_fields: data['fieldType'],
                    field_id: data['fieldId']
                },
                success: _initFromField
            });
        }
    };

    /**
     * Сохранение параметра
     * @param data {Object} Данные о параметре
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.saveParamData = function (data, id) {
        if (data && typeof data === 'object') {
            var ajaxData = {
                ajax: 1,
                action: 'create',
                button_id: this.buttonId,
                name: data['name'],
                field: data['field'],
                type_field: data['type_field'],
                default_value: {
                    is_field: data['default_value']['is_field'],
                    value: data['default_value']['value']
                },
                not_edit: data['not_edit'],
                required: data['required'],
            };
            if (typeof id !== 'undefined') {
                ajaxData['edit'] = id;
            }
            $.ajax({
                url: this.ajaxUrl,
                method: 'GET',
                data: ajaxData,
                async: false,
                success: function (xhr) {
                    if (xhr) {
                        if (typeof id === 'undefined') {
                            id = parseInt(JSON.parse(xhr));
                            this.renderNewParam(id, data);
                        } else {
                            this.changeParam(id);
                        }
                    }
                }.bind(this)
            });
        }
    };

    /**
     * Удаление параметра при buttonId != -1
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.dropParamAjax = function (id) {
        if (typeof id !== 'undefined') {
            $.ajax({
                url: this.ajaxUrl,
                method: 'GET',
                data: {
                    id: id,
                    action: 'delete',
                    button_id: this.buttonId,
                    ajax: 1
                }
            });
        }
    };

    /**
     * Получение данных для селекта поля `Значение по умолчанию`
     * @param fieldId {string|number} id выбранного поля
     */
    ParamsConstruct.prototype.getDefaultFieldData = function (fieldId) {
        if (typeof fieldId !== 'undefined') {
            $.ajax({
                url: this.ajaxUrl,
                async: false,
                method: 'GET',
                data: {
                    ajax: 1,
                    field_id: fieldId,
                    button_id: this.buttonId,
                    action: 'getListTypeValue'
                },
                success: _initDefaultSelectField
            })
        }
    };

    /** ==================== */

    /**
     * Инициализация событий
     */
    ParamsConstruct.prototype.initEvents = function () {
        var createParamBtn = $('#calc_params_create-btn');
        if (createParamBtn.length > 0) {
            createParamBtn.click(_onAddParamClick.bind(this));
        }
        var paramsLinks = $('.calc_param_link');
        if (paramsLinks.length > 0) {
            paramsLinks.each(function (i, link) {
                $(link).click(_onParamLinkClick.bind(this));
            }.bind(this));
        }
        var dropBtns = $('.cond-remove-btn');
        if (dropBtns.length > 0) {
            dropBtns.each(function (i, drop) {
                $(drop).click(_onDropParamClick.bind(this));
            }.bind(this));
        }
    };

    /**
     * Удаление параметра
     * @param id {string|number} id параметра
     */
    ParamsConstruct.prototype.dropParam = function (id) {
        var block = $('#calc_params_block_' + id);
        if (block.length > 0) {
            block.find($('.drop-btn')).off();
            block.find($('.calc_param_link')).off();
            block.remove();
            if (this.buttonId === -1) {
                delete this.paramsData[id];
            } else {
                this.dropParamAjax(id);
            }
        }
        var list = $('.calc_params_list');
        if (list.find('div.calc_params_block').length === 0) {
            $('.calc_no_params').show();
        }
    };

    /**
     * Инициализация
     */
    ParamsConstruct.prototype.init = function () {
        this.getFieldSaveData();
        this.initEvents();
    };

    window.ParamsConstruct = ParamsConstruct;

})();
