import { pick } from 'lodash';
/*
* Класс для преобразования серверных данных отчета
* в подготовленные данные для выгрузки в Excel
* для библиотеки SheetJS(xlsx)
*
* Пример данных:
* [
    { cell1:"Первый", cell2:"Второй" },
    { cell1: 1,  cell2: 2  },
    { cell1: 3,  cell2: 4  },
    { cell1: 5,  cell2: 6  }
  ]
*/

class NormalizationJsonToExcel {
    constructor(inputData) {
        this.data = inputData;
    }

    /**
     * Интерфейс класса для взаимодействия
     */

    // Геттер названия
    get getExcelFileName() {
        const fileName = this.data.report.name;
        return fileName;
    }

    // Вызов нормализации данных
    normalizeData() {
        let outputArrayOfData = []; // На выходе нужен массив для SheetJS
        const { layout, data } = this.data;

        // Нормализуем все блоки в отчете
        layout.forEach( value => {
            const blockType = value.type;
            switch (blockType) { // Тип layout'а
                case "table":
                    let tableArrayData = [];
                    /* Работа с названием таблицы */
                    const thisTableName = this.getTableName(value.caption);
                    tableArrayData.push(thisTableName);

                    /* Работа с шапкой таблицы */
                    const thisTableHead = this.getTableHeaders(value.body, value.head);
                    tableArrayData.push(thisTableHead);

                    /* Работа с телом таблицы */
                    const dataKey = value.data; // Ключ для данных
                    const thisTableData = data[dataKey];

                    // Прогоняем через фильтр для специальных поле (например поля связи, в будущем изображения и файлы)
                    const thisTableBody = this.refactorTableBody(thisTableData, thisTableHead);
                    
                    for (let i in thisTableBody) {
                        // Не добавляем дополнительную информацию
                        if ( thisTableBody[i] instanceof Object )
                            tableArrayData.push(thisTableBody[i]);
                    }

                    /* Работа с подвалом таблицы */
                    const thisTableFooter = this.getTableFooter(value.foot, data);
                    tableArrayData.push(thisTableFooter);

                    outputArrayOfData.push(tableArrayData);
                    break;
                case "block":
                    /* Работа с контентом показателя */
                    const blockContent = value.layout;
                    let blockArrayData = [];
                    blockContent.forEach( blockValue => { // cell1 - жесткое название первой ячейки
                        let blockContent = '';
                        if (typeof blockValue.content != 'undefined') { // Если есть prop content
                            blockContent = blockValue.content;
                        } else {
                            blockContent = blockValue.layout[0].content;
                        }
                        // Проверим есть ли значение по ключу в данных
                        const clearKey = blockContent.replace(/~/g, "");
                        if ( typeof data[clearKey] != 'undefined' ) // Если в данных есть значение, заменяем текущее
                            blockContent = data[clearKey];
                        blockArrayData.push({
                            "cell1": blockContent
                        })
                    });

                    outputArrayOfData.push(blockArrayData);
                    break;
                default:
                    break;
            }
        });

        return outputArrayOfData;
    }

    /**
     * Внутренние методы
     */

    // Берет названия таблиц
    getTableName(tName) {
        const nameObj = { // cell1 - жесткое название первой ячейки
            'cell1' : tName
        };

        return nameObj;
    }

    // Берет хедер таблиц
    getTableHeaders(bodyArr, headArr) {
        if ( bodyArr instanceof Array && headArr instanceof Array) {
            let headerObj = {};

            bodyArr.forEach( (value, index) => {
                if ( typeof value === 'string') { // Без ссылок (чистые значения)
                    const cleanValue = value.replace(/~/g,""); // Удаляем доп символ ~
                    const headerVal = headArr[index];
                    headerObj[cleanValue] = headerVal;
                } else { // С ссылками в таблице
                    const cleanValue = value.layout[0].content.replace(/~/g,""); // Удаляем доп символ ~
                    const headerVal = headArr[index];
                    headerObj[cleanValue] = headerVal;
                }
            });

            return headerObj;
        } else
            return {};
    }

    // Преобразования тела таблицы для специальных поле (например поля связи, в будущем изображения и файлы)
    refactorTableBody(inputBody, tableHead) {
        const headerKeys = Object.keys(tableHead);
        let outputBody = {...inputBody};
        let filteredBody = {};
        for (let i in outputBody) {
            filteredBody[i] = {...pick(outputBody[i], headerKeys)}; // Фильтруем данные исходя из значений в хедере
        }

        for (let i in filteredBody) {
            for (let j in filteredBody[i]) {
                let cellVal = filteredBody[i][j];
                if ( typeof cellVal != 'string' ) { // Если значение ячейки таблицы не строка, тогда берем значение из __REPCON_ARRAY_DEFAULT_VALUE__
                    if ( cellVal instanceof Object && cellVal['__REPCON_ARRAY_DEFAULT_VALUE__']) {
                        cellVal = cellVal['__REPCON_ARRAY_DEFAULT_VALUE__'];
                    } else {
                        cellVal = '';
                    }

                    filteredBody[i][j] = cellVal;
                }
            }
        }

        return filteredBody;
    }

    // Берет футер таблиц
    getTableFooter(footerArr, tableData) {
        if ( footerArr instanceof Array) {
            if (footerArr.length === 0) return {}; // Проверяем футер на пустоту
            let footerObj = {};

            footerArr.forEach( (value, index) => {
                let footerVal = '';
                if ( typeof value === 'string') { // Без ссылок (чистые значения)
                    const cleanValue = value.replace(/~/g,""); // Удаляем доп символ ~
                    footerVal = tableData[cleanValue];
                } else { // С ссылками в таблице
                    const cleanValue = value.content.replace(/~/g,""); // Удаляем доп символ ~
                    footerVal = tableData[cleanValue];
                }

                const cellKey = index + 1;
                footerObj['cell' + cellKey] = footerVal || '';
            });

            return footerObj;
        } else
            return {};
    }
}

export default NormalizationJsonToExcel;